<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;

class StoreAction extends Model
{
    use HasFactory;

    protected $fillable = [
        'user_id',
        'server_id',
        'transaction_id',
        'package_id',
        'actions',
        'executed',
        'received_at',
    ];

    protected $casts = [
        'actions' => 'array',
        'executed' => 'boolean',
        'received_at' => 'datetime',
    ];

    /**
     * Get the user that owns the store action.
     */
    public function user(): BelongsTo
    {
        return $this->belongsTo(User::class);
    }

    /**
     * Get the server that the store action belongs to.
     */
    public function server(): BelongsTo
    {
        return $this->belongsTo(Server::class);
    }

    /**
     * Get the transaction that created this store action.
     */
    public function transaction(): BelongsTo
    {
        return $this->belongsTo(Transaction::class);
    }

    /**
     * Get the package that this store action is for.
     */
    public function package(): BelongsTo
    {
        return $this->belongsTo(Package::class);
    }

    /**
     * Scope to get only unexecuted actions.
     */
    public function scopeUnexecuted($query)
    {
        return $query->where('executed', false);
    }

    /**
     * Scope to get actions for a specific server.
     */
    public function scopeForServer($query, $serverId)
    {
        return $query->where('server_id', $serverId);
    }

    /**
     * Scope to get actions for a specific user.
     */
    public function scopeForUser($query, $userId)
    {
        return $query->where('user_id', $userId);
    }
}